﻿/**********************************************************************************
 *
 * File:			AddMetadata.c
 *
 * Usage:			AddMetadata <inputPath> <outputPath> [<mdatafile>]
 *					Example: AddMetadata in.pdf out.pdf MetadataRef.pdf
 *
 * Title:			Add Metadata to PDF
 *
 * Description:		Set metadata such as author, title, and creator of a PDF
 *					document. Optionally use the metadata of another PDF
 *					document or the content of an XMP file.
 *
 * Author:			株式会社トラスト・ソフトウェア・システム
 *
 * Copyright:		Copyright (C) 2025 株式会社トラスト・ソフトウェア・システム
 *
 *********************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "PdfTools.h"

#include <locale.h>
#if !defined(_TCHAR_DEFINED)
#define TCHAR char
#define _tcslen strlen
#define _tcscat strcat
#define _tcscpy strcpy
#define _tcsrchr strrchr
#define _tcstok strtok
#define _tcslen strlen
#define _tcscmp strcmp
#define _tcsftime strftime
#define _tcsncpy strncpy
#define _tmain main
#define _tfopen_s fopen_s
#define _ftprintf fprintf
#define _stprintf sprintf
#define _tstof atof
#define _tremove remove
#define _tprintf printf
#define _T(str) str
#endif


#define MIN(a, b)		(((a) < (b) ? (a) : (b)))
#define ARRAY_SIZE(a)	(sizeof(a) / sizeof(a)[0])

#define GOTO_CLEANUP_IF_NULL_PRINT_ERROR(inVar, ...)											\
	do																							\
	{																							\
		if ((inVar) == NULL)																	\
		{																						\
			nBufSize = PdfTools_GetLastErrorMessage(NULL, 0);									\
			PdfTools_GetLastErrorMessage(szErrorBuff, MIN(ARRAY_SIZE(szErrorBuff), nBufSize));	\
			_tprintf(__VA_ARGS__);																\
			iRet = 1;																			\
			goto cleanup;																		\
		}																						\
	} while (0);

#define GOTO_CLEANUP_IF_FALSE_PRINT_ERROR(outBool, ...)											\
	do																							\
	{																							\
		if ((outBool) == FALSE)																	\
		{																						\
			nBufSize = PdfTools_GetLastErrorMessage(NULL, 0);									\
			PdfTools_GetLastErrorMessage(szErrorBuff, MIN(ARRAY_SIZE(szErrorBuff), nBufSize));	\
			_tprintf(__VA_ARGS__);																\
			iRet = 1;																			\
			goto cleanup;																		\
		}																						\
	} while (0);

int Usage()
{
	printf("Usage: AddMetadata <inputPath> <outputPath> [<mDataFile>].\n");
	printf("return codes:\n");
	printf("	0: Successful completion\n");
	printf("	1: Execution failed\n");

	return 1;
}

int _tmain(int argc, TCHAR* argv[])
{
	int		iRet = 0;
	size_t	nBufSize;
	TCHAR	szErrorBuff[1024];

	setlocale(LC_ALL, "Japanese");

	// 引数チェック
	if (argc < 3 || argc > 4)
	{
		return Usage();
	}

	TCHAR* szInPath = argv[1];							// /absolute/path/to/input.pdf
	TCHAR* szOutPath = argv[2];							// /absolute/path/to/output.pdf
	TCHAR* szMetaPath = argc == 4 ? argv[3] : _T("");	// /absolute/path/to/metaData.pdf
	FILE* pInStream = NULL;
	FILE* pOutStream = NULL;
	FILE* pMetaStream = NULL;		// メタデータを参照するPDFのStream
	TPdfToolsPdf_Document* pInDoc = NULL;

#define BUFFER_SIZE	256
	TCHAR buffer[BUFFER_SIZE];

	// Initialize library
	PdfTools_Initialize();

	// 既定では、テスト用ライセンスが有効です。ただし、結果に「透かし」が追加されます。
	// ライセンスキーを取得している場合は、以下のラインのコメントを外しライセンスキーをセットしてください。
	// GOTO_CLEANUP_IF_FALSE_PRINT_ERROR(PdfTools_Sdk_Initialize(_T("ここにライセンスキーをセット"), NULL),
	// 									_T("Failed to set the license key. %s (ErrorCode: 0x%08x).\n"), szErrorBuff,
	//									PdfTools_GetLastError());

	// 入力ファイルを開く
	_tfopen_s(&pInStream, szInPath, _T("rb"));
	GOTO_CLEANUP_IF_NULL_PRINT_ERROR(pInStream, _T("Failed to open the input file \"%s\" for reading.\n"), szInPath);
	TPdfToolsSys_StreamDescriptor inDesc;
	PdfToolsSysCreateFILEStreamDescriptor(&inDesc, pInStream, 0);
	pInDoc = PdfToolsPdf_Document_Open(&inDesc, _T(""));
	GOTO_CLEANUP_IF_NULL_PRINT_ERROR(
		pInDoc, _T("Failed to create a document from the input file \"%s\". %s (ErrorCode: 0x%08x).\n"), szInPath,
		szErrorBuff, PdfTools_GetLastError());

	TPdfToolsPdf_MetadataSettings* metaSet = PdfToolsPdf_MetadataSettings_New();
	TPdfToolsPdf_Document* metaDoc = NULL;

	if (szMetaPath) {
		// 参照するPDF
		if (!_tfopen_s(&pMetaStream, szMetaPath, _T("rb"))) {
			TPdfToolsSys_StreamDescriptor metaDesc;
			PdfToolsSysCreateFILEStreamDescriptor(&metaDesc, pMetaStream, 0);
			metaDoc = PdfToolsPdf_Document_Open(&metaDesc, NULL);
			if (metaDoc) {
				TPdfToolsPdf_Metadata* meta = PdfToolsPdf_Document_GetMetadata(metaDoc);
				if (meta) {
					if (PdfToolsPdf_Metadata_GetTitle(meta, buffer, BUFFER_SIZE)) PdfToolsPdf_MetadataSettings_SetTitle(metaSet, buffer);
					if (PdfToolsPdf_Metadata_GetAuthor(meta, buffer, BUFFER_SIZE)) PdfToolsPdf_MetadataSettings_SetAuthor(metaSet, buffer);
					if (PdfToolsPdf_Metadata_GetSubject(meta, buffer, BUFFER_SIZE)) PdfToolsPdf_MetadataSettings_SetSubject(metaSet, buffer);
					if (PdfToolsPdf_Metadata_GetKeywords(meta, buffer, BUFFER_SIZE)) PdfToolsPdf_MetadataSettings_SetKeywords(metaSet, buffer);
					if (PdfToolsPdf_Metadata_GetCreator(meta, buffer, BUFFER_SIZE)) PdfToolsPdf_MetadataSettings_SetCreator(metaSet, buffer);
					if (PdfToolsPdf_Metadata_GetProducer(meta, buffer, BUFFER_SIZE)) PdfToolsPdf_MetadataSettings_SetProducer(metaSet, buffer);
					TPdfToolsSys_Date date;
					if (PdfToolsPdf_Metadata_GetCreationDate(meta, &date)) PdfToolsPdf_MetadataSettings_SetCreationDate(metaSet, &date);
					if (PdfToolsPdf_Metadata_GetModificationDate(meta, &date)) PdfToolsPdf_MetadataSettings_SetModificationDate(metaSet, &date);
				}
				else goto Local_Set;
			}
			else goto Local_Set;
		}
		else goto Local_Set;
	}
	else {
Local_Set:
		PdfToolsPdf_MetadataSettings_SetTitle(metaSet, _T("Title"));
		PdfToolsPdf_MetadataSettings_SetAuthor(metaSet, _T("someAuthor"));
		PdfToolsPdf_MetadataSettings_SetSubject(metaSet, _T("Subject"));
		PdfToolsPdf_MetadataSettings_SetCreator(metaSet, _T("someCreator"));
		PdfToolsPdf_MetadataSettings_SetProducer(metaSet, _T("someProducer"));
	}

	// 参照PDFを閉じる
	if (metaDoc) PdfToolsPdf_Document_Close(metaDoc);
	if (pMetaStream) fclose(pMetaStream);

	// 出力Streamを作成
	_tfopen_s(&pOutStream, szOutPath, _T("wb+"));
	GOTO_CLEANUP_IF_NULL_PRINT_ERROR(pOutStream, _T("Failed to create output file \"%s\" for writing.\n"), szOutPath);
	TPdfToolsSys_StreamDescriptor outDesc;
	PdfToolsSysCreateFILEStreamDescriptor(&outDesc, pOutStream, 0);

	// 出力オプション
	TPdfToolsPdf_OutputOptions* outOpt = PdfToolsPdf_OutputOptions_New();
	PdfToolsPdf_OutputOptions_SetMetadataSettings(outOpt, metaSet);

	// 出力
	TPdfToolsOptimization_Optimizer* opt = PdfToolsOptimization_Optimizer_New();
	PdfToolsOptimization_Optimizer_OptimizeDocument(opt, pInDoc, &outDesc,
		(TPdfToolsOptimizationProfiles_Profile*)PdfToolsOptimizationProfiles_MinimalFileSize_New(), outOpt);

cleanup:
	if (pInDoc) PdfToolsPdf_Document_Close(pInDoc);
	if (pInStream) fclose(pInStream);
	if (pOutStream) fclose(pOutStream);

	PdfTools_Uninitialize();

	return iRet;
}

